/*
 * Copyright (c) 2006-2025 Progress Software Corporation and/or its subsidiaries or affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.marklogic.xcc.impl.handlers;

import com.fasterxml.jackson.databind.JsonNode;
import com.marklogic.http.HttpChannel;
import com.marklogic.xcc.Request;
import com.marklogic.xcc.exceptions.PDCloudRequestException;
import com.marklogic.xcc.impl.Credentials;
import com.marklogic.xcc.impl.PDCloudAuthManager;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.logging.Level;
import java.util.logging.Logger;

public class PDCloudSuccessResponseHandler implements ResponseHandler {
    @Override
    public Object handleResponse(HttpChannel http, int responseCode,
                                 Request request, Object attachment,
                                 Logger logger) throws IOException,
        PDCloudRequestException {
        String contentType = http.getResponseContentType();
        if (contentType.equals("application/json")) {
            JsonNode jsonMap = http.getResponseJsonBody();
            String sessionToken = jsonMap.get("access_token").textValue();
            String expiration = jsonMap.get(".expires").textValue();
            long duration = jsonMap.get("expires_in").longValue();
            try {
                if (attachment instanceof Credentials.PDCloudAuthConfig) {
                    PDCloudAuthManager.setTokenConfig(
                        ((Credentials.PDCloudAuthConfig)attachment).getApiKey(),
                        sessionToken.toCharArray(),
                        tokenExpirationInMillis(expiration),
                        duration);
                    if (logger.isLoggable(Level.FINE)) {
                        logger.log(Level.FINE, "Successfully obtained session" +
                            "token from MarkLogic Cloud token endpoint.");
                    }
                }
            } catch (ParseException e) {
                throw new PDCloudRequestException("Exception parsing " +
                    "Progress Data Cloud response body. " + e.getMessage(), request,
                    false);
            }
        } else {
            throw new PDCloudRequestException("Malformed response body from " +
                "Progress Data Cloud. Please report to Customer Support.", request,
                false);
        }
        return (null);
    }

    // Convert session token expiration time to milliseconds
    private long tokenExpirationInMillis(String dateString)
        throws ParseException {
        SimpleDateFormat sdf = new SimpleDateFormat(
            "EEE, dd MMM yyyy HH:mm:ss z");
        Date date = sdf.parse(dateString);
        return date.getTime();
    }
}
