/*
 * Copyright (c) 2006-2025 Progress Software Corporation and/or its subsidiaries or affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.marklogic.xcc.types.impl;

import com.marklogic.xcc.types.ValueType;
import com.marklogic.xcc.types.VecVector;

public class VecVectorImpl extends AbstractStringItem implements VecVector {
    private final float[] values;

    public VecVectorImpl(String value) throws IllegalArgumentException {
        super(ValueType.VEC_VECTOR, value);
        if(value == null) {
            throw new IllegalArgumentException("Value string must not be null.");
        }
        if(value.isEmpty()) {
            throw new IllegalArgumentException("Value string must not be empty.");
        }
        if(!isVectorFormat(value)) {
            throwVectorFormatError();
        }
        values = parseVectorString(value);
    }

    public VecVectorImpl(float[] values) throws IllegalArgumentException {
        super(ValueType.VEC_VECTOR, null);
        if(values == null) {
            throw new IllegalArgumentException("Array must not be null.");
        }
        if(values.length < 1) {
            throw new IllegalArgumentException("Array must not be empty.");
        }

        this.values = values;
    }

    private static boolean isVectorFormat(String str) {
        // just need to check if first and last character align with expectations
        // error checking within string will occur during parseVectorString
        return str.charAt(0) == '[' && str.charAt(str.length() - 1) == ']';
    }

    private static void throwVectorFormatError() throws IllegalArgumentException {
        throw new IllegalArgumentException("Value string must be in vector format. Example: [ x, y, z ]");
    }

    private static void throwNumberFormatAtIndexError(String val, int index) throws NumberFormatException {
        throw new NumberFormatException("Value [" + val + "] at index [" + index + "] can not be parsed");
    }

    private static float[] parseVectorString(String vectorString) throws IllegalArgumentException {
        // There will be one more value in the string than there are commas
        int valueCount = countCommas(vectorString) + 1;
        float[] result = new float[valueCount];
        StringBuilder input = new StringBuilder();

        int arrayIndex = 0;
        char c;

        // start at 1 since at this point, we know 0 is a '['
        for(int i = 1; i < vectorString.length(); ++i) {
            c = vectorString.charAt(i);
            switch(c) {
                case '[':
                    // we encountered a second vector start, invalid formated string
                    throwVectorFormatError();
                case ' ': // skip whitespace
                    continue;
                case ']':
                    if(i != vectorString.length() - 1) {
                        // we encountered a pre-mature end of vector, invalid format string
                        throwVectorFormatError();
                    }
                    // no error with ']', allowing pass through to ',' case
                case ',':
                    // end of current value, process current float
                    String val = input.toString();
                    if(val.isEmpty()) {
                        throw new IllegalArgumentException("Value missing at index [" + arrayIndex + "]");
                    }
                    try {
                        result[arrayIndex] = Float.parseFloat(val);
                    } catch (NumberFormatException e) {
                        // Make parsing error more detailed, by providing array index it occurred at
                        throwNumberFormatAtIndexError(val, arrayIndex);
                    }
                    ++arrayIndex;
                    input.setLength(0); // clear input for next round
                    break;
                default:
                    // record character for parsing
                    input.append(c);
                    break;
            }
        }

        return result;
    }

    private static int countCommas(String str) {
        int result = 0;
        for (int i = 0; i < str.length(); ++i) {
            if(str.charAt(i) == ',') ++result;
        }
        return result;
    }

    @Override
    public String asString() {
        if (value == null) {
            StringBuilder sb = new StringBuilder();
            sb.append("[ ");
            for (Float v : values) {
                sb.append(v.floatValue());
                sb.append(", ");
            }
            value = sb.substring(0, sb.length()-2) + " ]";
        }
        return value;
    }

    @Override
    public float[] getValues() {
        return values;
    }
}
